document.addEventListener('DOMContentLoaded', function () {

    const scanSound = new Audio('/public/assets/sounds/beep.mp3');

    /* =========================================================
       HELPER
    ========================================================= */

    function postJSON(url, data) {
        return fetch(url, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            credentials: 'same-origin',
            body: JSON.stringify(data)
        }).then(res => res.json());
    }

    function formatRM(value) {
        return 'RM ' + parseFloat(value).toFixed(2);
    }

/* =========================================================
   PROFESSIONAL ERROR MODAL
========================================================= */

function showErrorModal(message) {
    const modal = document.getElementById('errorModal');
    const msg = document.getElementById('errorMessage');

    if (!modal || !msg) return;

    msg.innerText = message;
    modal.classList.remove('hidden');
}

document.getElementById('closeErrorModal')?.addEventListener('click', function () {
    document.getElementById('errorModal')?.classList.add('hidden');
});

    /* =========================================================
       ACTION BUTTON VISIBILITY
    ========================================================= */

function updateActionButtons() {

    const cancelBtn = document.getElementById('cancelBtn');
    const suspendBtn = document.getElementById('suspendBtn');
    const actionTop = document.querySelector('.action-top');

    const hasItems = !document.querySelector('#cartBody .cart-empty');

    const hasCustomer =
        !!document.querySelector('.customer-profile') ||
        !!document.querySelector('.customer-profile-premium');

    if (cancelBtn) {
        cancelBtn.classList.toggle('hidden-btn', !hasItems);
    }

    if (suspendBtn) {
        suspendBtn.classList.toggle('hidden-btn', !(hasItems && hasCustomer));
    }

    if (actionTop) {
        actionTop.classList.toggle('only-menu', !hasItems);
    }
}

/* =========================================================
   ACTION DROPDOWN TOGGLE
========================================================= */

const actionBtn = document.getElementById('actionMenuBtn');
const dropdown = document.getElementById('actionDropdown');

actionBtn?.addEventListener('click', function (e) {
    e.stopPropagation();
    dropdown.classList.toggle('show');
});

document.addEventListener('click', function () {
    dropdown?.classList.remove('show');
});

    /* =========================================================
       BARCODE SCAN
    ========================================================= */

    const barcodeInput = document.getElementById('barcode-input');

    if (barcodeInput) {
        barcodeInput.addEventListener('keydown', function (e) {

            if (e.key !== 'Enter') return;

            e.preventDefault();

            const code = this.value.trim();
            if (!code) return;

            postJSON('/pos/scan-ajax', { barcode: code })
                .then(res => {
    if (res.error) {
        showErrorModal(res.error);
        return;
    }
    renderCart(res);
})

                .catch(console.error);

            scanSound.play();
            this.value = '';
        });
    }

    /* =========================================================
       RENDER CART
    ========================================================= */

    function renderCart(cart) {

        if (!cart || !cart.items) return;

        const body = document.getElementById('cartBody');
        const subtotal = document.getElementById('cartSubtotal');
        const total = document.getElementById('cartTotal');
        const due = document.getElementById('cartDue');
        const paymentSection = document.getElementById('paymentSection');

        if (!body) return;

        body.innerHTML = '';

        const hasItems = cart.items.length > 0;

        if (!hasItems) {

            body.innerHTML = `
                <div class="cart-empty">
                    There are no items in the cart <strong>[Sales]</strong>
                </div>
            `;

            if (paymentSection) paymentSection.style.display = 'none';

        } else {

            if (paymentSection) paymentSection.style.display = 'block';

            cart.items.forEach((item, i) => {

                const wrapper = document.createElement('div');
                wrapper.className = 'cart-item';

                wrapper.innerHTML = `
                    <div class="cart-row">
                        <div>
                            <button type="button" class="remove-item" data-index="${i}">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>

                        <div class="item-name">${item.name}</div>

                        <div>${parseFloat(item.price).toFixed(2)}</div>

                        <div>
                            <button type="button" class="qty-btn" data-action="dec" data-index="${i}">−</button>
                            <span class="qty">${item.qty}</span>
                            <button type="button" class="qty-btn" data-action="inc" data-index="${i}">+</button>
                        </div>

                        <div>${(item.price * item.qty).toFixed(2)}</div>
                    </div>

                    <div class="cart-row-details">
                        <div><strong>Tier</strong> ${item.tier ?? 'None'}</div>
                        <div><strong>Category</strong> ${item.category ?? '-'}</div>
                        <div><strong>UPC/EAN/ISBN</strong> ${item.upc ?? '-'}</div>
                        <div><strong>Stock</strong> ${item.stock ?? '-'}</div>
                        <div><strong>Tax</strong> <a href="#">Edit Taxes</a></div>
                    </div>
                `;

                body.appendChild(wrapper);
            });

            autoSelectCash();
        }

        const formatted = formatRM(cart.total);

        if (subtotal) subtotal.innerText = formatted;
        if (total) total.innerText = formatted;
        if (due) due.innerText = formatted;

        updateActionButtons();
    }

    /* =========================================================
       AUTO SELECT CASH
    ========================================================= */

    function autoSelectCash() {

        const cashBtn = document.querySelector('.pay-btn[data-method="cash"]');
        const cashBox = document.getElementById('cashBox');

        if (!cashBtn) return;

        if (!document.querySelector('.pay-btn.active')) {
            cashBtn.classList.add('active');
            if (cashBox) cashBox.style.display = 'block';
        }
    }

    /* =========================================================
       GLOBAL CLICK
    ========================================================= */

    document.addEventListener('click', function (e) {

        const target = e.target.closest('button');
        if (!target) return;

if (target.classList.contains('qty-btn')) {

    postJSON('/pos/update-qty-ajax', {
        index: target.dataset.index,
        action: target.dataset.action
    })
    .then(res => {

        if (res.error) {
            showErrorModal(res.error);
            return;
        }

        renderCart(res);
    });
    
        }

        if (target.classList.contains('remove-item')) {
            postJSON('/pos/remove-item-ajax', {
                index: target.dataset.index
            }).then(renderCart);
        }

        if (target.classList.contains('pay-btn')) {

            document.querySelectorAll('.pay-btn')
                .forEach(b => b.classList.remove('active'));

            target.classList.add('active');

            const cashBox = document.getElementById('cashBox');

            if (cashBox) {
                cashBox.style.display =
                    target.dataset.method === 'cash' ? 'block' : 'none';
            }
        }
    });

    /* =========================================================
       SALES PERSON
    ========================================================= */

    const currentSP = document.getElementById('currentSalesPerson');
    const spList = document.getElementById('salesPersonList');

    if (currentSP && spList) {

        currentSP.addEventListener('click', function (e) {
            e.stopPropagation();
            spList.classList.toggle('show');
        });

        document.querySelectorAll('.sales-person-option')
            .forEach(option => {

                option.addEventListener('click', function (e) {
                    e.stopPropagation();

                    const id = this.dataset.id;

                    postJSON('/pos/change-salesperson', { id: id })
                        .then(res => {

                            if (!res.success) return;

                            currentSP.innerText = this.innerText;

                            document.querySelectorAll('.sales-person-option')
                                .forEach(o => o.classList.remove('active'));

                            this.classList.add('active');
                            spList.classList.remove('show');
                        });
                });
            });

        document.addEventListener('click', function () {
            spList.classList.remove('show');
        });
    }

/* =========================================================
   COMPLETE SALE
========================================================= */

const checkoutBtn = document.getElementById('checkoutBtn');

if (checkoutBtn) {
    checkoutBtn.addEventListener('click', function () {

        const activePayment = document.querySelector('.pay-btn.active');
        const cashInput = document.getElementById('cashTendered');

        if (!activePayment) {
            alert('Please select payment method');
            return;
        }

        const paymentMethod = activePayment.dataset.method;
        const amountTendered = paymentMethod === 'cash'
            ? parseFloat(cashInput?.value || 0)
            : null;

        checkoutBtn.disabled = true;
        checkoutBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';

        const transactionUUID = crypto.randomUUID();

postJSON('/pos/checkout', {
    payment_method: paymentMethod,
    amount_tendered: amountTendered,
    transaction_uuid: transactionUUID,
    token: document.getElementById('checkoutToken').value
})


        .then(res => {

if (!res.success) {

    showErrorModal(res.message || 'Checkout failed');

    checkoutBtn.disabled = false;
    checkoutBtn.innerHTML = 'Complete Sale';
    return;
}



            window.location.href = '/pos/receipt/' + res.sale_id;

        })
        .catch(err => {
            console.error(err);
            alert('Checkout error');
            checkoutBtn.disabled = false;
            checkoutBtn.innerHTML = 'Complete Sale';
        });
    });
}

    /* =========================================================
       CUSTOMER SEARCH
    ========================================================= */

    const customerSearch = document.getElementById('customerSearch');
    const customerDropdown = document.getElementById('customerDropdown');

    if (customerSearch && customerDropdown) {

        customerSearch.addEventListener('keyup', function () {

            const keyword = this.value.trim();

            if (keyword.length < 2) {
                customerDropdown.innerHTML = '';
                customerDropdown.classList.remove('show');
                return;
            }

            fetch('/customers/ajax-search?q=' + encodeURIComponent(keyword))
                .then(res => res.json())
                .then(data => {

                    customerDropdown.innerHTML = '';

                    if (!data.length) {
                        customerDropdown.classList.remove('show');
                        return;
                    }

                    data.forEach(c => {

                        const div = document.createElement('div');
                        div.className = 'customer-option';
                        div.textContent = `${c.name} (${c.phone})`;

                        div.addEventListener('click', function () {
                            postJSON('/pos/select-customer', { id: c.id })
                                .then(() => location.reload());
                        });

                        customerDropdown.appendChild(div);
                    });

                    customerDropdown.classList.add('show');
                })
                .catch(console.error);
        });

        document.addEventListener('click', function (e) {
            if (!customerSearch.closest('.customer-search-modern')?.contains(e.target)) {
                customerDropdown.classList.remove('show');
            }
        });
    }

    /* =========================================================
       QUICK ADD PRODUCT
    ========================================================= */
    
document.querySelector('.add-product')?.addEventListener('click', function () {

    window.location.href = '/inventory/create?return=pos';

});

    /* =========================================================
       DETACH CUSTOMER
    ========================================================= */
    
document.querySelector('.detach-btn')?.addEventListener('click', function(){
    postJSON('/pos/remove-customer',{})
        .then(()=>location.reload());
});

    /* =========================================================
       SET MODE
    ========================================================= */
    
document.querySelector('.sale-mode')?.addEventListener('click', function(){

    const mode = prompt('Enter mode: sale / rma / estimate');
    if (!mode) return;

    postJSON('/pos/set-mode', { mode: mode.toLowerCase() })
        .then(() => alert('Mode changed'));

});

    /* =========================================================
       PRODUCT PICKER
    ========================================================= */
document.querySelector('.show-grid')?.addEventListener('click', function(){

    fetch('/pos/products-grid')
        .then(res=>res.json())
        .then(products=>{
            const name = prompt(
                products.map(p=>p.id+' - '+p.name).join('\n')
            );

            if (!name) return;

            postJSON('/pos/scan-ajax',{barcode:name})
                .then(renderCart);
        });

});

    /* =========================================================
       CANCEL SALE BUTTON
    ========================================================= */
document.getElementById('cancelBtn')?.addEventListener('click', function(){

    if (!confirm('Cancel entire sale?')) return;

    postJSON('/pos/cancel-sale',{})
        .then(()=>location.reload());

});

/* =========================================================
   SUSPEND SALE BUTTON
========================================================= */

document.getElementById('suspendBtn')?.addEventListener('click', function(){

    const note = prompt('Suspend note (e.g. Customer coming back later)');
    if (note === null) return;

    postJSON('/pos/suspend-sale', { note: note })
        .then(res=>{
            if(res.success){
                alert('Sale suspended as ' + res.reference);
                location.reload();
            }
        });

});

/* =========================================================
   SHOW SUSPENDED MODAL
========================================================= */

const suspendModal = document.getElementById('suspendModal');
const suspendTableBody = document.getElementById('suspendTableBody');
const suspendSearch = document.getElementById('suspendSearch');

document.querySelector('.show-suspended')?.addEventListener('click', function(){

    fetch('/pos/suspended-list')
        .then(res => res.json())
        .then(rows => {

            if(!rows.length){
                alert('No suspended sales found');
                return;
            }

            suspendTableBody.innerHTML = '';

            rows.forEach(r => {

                const tr = document.createElement('tr');

                tr.innerHTML = `
                    <td>${r.reference_no}</td>
                    <td>${r.customer_name ?? 'Walk-in'}</td>
                    <td>${r.salesperson_name ?? '-'}</td>
                    <td>RM ${parseFloat(r.total).toFixed(2)}</td>
                    <td>${r.note ?? '-'}</td>
                    <td>${r.created_at}</td>
                    <td>
                        <button class="restore-btn" data-id="${r.id}">Restore</button>
                        <button class="cancel-suspend-btn" data-id="${r.id}">Cancel</button>
                    </td>
                `;

                suspendTableBody.appendChild(tr);
            });

            suspendModal.classList.remove('hidden');
        });
});

/* CLOSE */
document.getElementById('closeSuspendModal')?.addEventListener('click', function(){
    suspendModal.classList.add('hidden');
});

/* SEARCH */
suspendSearch?.addEventListener('keyup', function(){

    const keyword = this.value.toLowerCase();

    document.querySelectorAll('#suspendTableBody tr').forEach(row => {
        row.style.display =
            row.innerText.toLowerCase().includes(keyword)
            ? ''
            : 'none';
    });
});

/* RESTORE + CANCEL */
document.addEventListener('click', function(e){

    if(e.target.classList.contains('restore-btn')){

        postJSON('/pos/restore-suspended', {
            id: e.target.dataset.id
        }).then(res=>{
            if(res.success){
                suspendModal.classList.add('hidden');
                location.reload();
            }
        });
    }

    if(e.target.classList.contains('cancel-suspend-btn')){

        if(!confirm('Cancel this suspended sale?')) return;

        postJSON('/pos/cancel-suspended', {
            id: e.target.dataset.id
        }).then(()=>{
            e.target.closest('tr').remove();
        });
    }

});


    /* =========================================================
       INITIAL LOAD FIX
    ========================================================= */

    updateActionButtons();
    if (window.initialCart) {
    renderCart(window.initialCart);
}

});
