/*
|--------------------------------------------------------------------------
| STOCK RULE
|--------------------------------------------------------------------------
| All stock changes MUST go through this service.
| Never update product_stocks directly anywhere else.
| This ensures consistency, logging and race-condition safety.
|--------------------------------------------------------------------------
*/

<?php
namespace App\POS;

use App\Core\Database;
use App\Inventory\StockService;
use App\POS\PosAuditService;

class PosService
{
    public function checkout(
    array $cart,
    $customer = null,
    $paymentMethod = null,
    $tendered = null,
    $changeAmount = 0,
    string $transactionUUID = null
)

    {
        if (empty($cart['items'])) {
            return false;
        }

        $branchId     = $_SESSION['branch_id'];
        $suspendId    = $_SESSION['restored_suspend_id'] ?? null; 
        $stockService = new StockService();

        Database::begin();

$audit = new PosAuditService();

        try {
            
            $audit->log(
    'checkout_started',
    null,
    null,
    'Checkout initiated | Branch: '.$branchId.' | Total: '.$cart['total']
);

// 🚫 Prevent duplicate transaction
$existing = Database::fetchOne(
    "SELECT id FROM pos_sales WHERE transaction_uuid = ?",
    [$transactionUUID]
);

if ($existing) {
    $audit->log(
        'checkout_duplicate_uuid',
        $existing['id'],
        'sale',
        'Duplicate transaction attempt'
    );

    throw new \Exception("Duplicate transaction detected.");
}

            /* ===============================
               1️⃣ CREATE SALE
            =============================== */
Database::execute(
    "INSERT INTO pos_sales 
    (transaction_uuid, branch_id, customer_id, total, payment_method, tendered, change_amount, created_at)
    VALUES (?, ?, ?, ?, ?, ?, ?, NOW())",
    [
        $transactionUUID,
        $branchId,
        $customer['id'] ?? null,
        $cart['total'],
        $paymentMethod,
        $tendered,
        $changeAmount
    ]
);


            $saleId = Database::lastInsertId();

            if (!$saleId) {
                throw new \Exception('Sale insert failed');
            }

            /* ===============================
               2️⃣ PROCESS EACH ITEM
            =============================== */
            foreach ($cart['items'] as $item) {

                Database::execute(
                    "INSERT INTO pos_sale_items
                    (pos_sale_id, product_id, quantity, price, branch_id)
                    VALUES (?, ?, ?, ?, ?)",
                    [
                        $saleId,
                        $item['id'],
                        $item['qty'],
                        $item['price'],
                        $branchId
                    ]
                );

                // Skip manual / custom items
                if ($item['id'] <= 0) {
                    continue;
                }

                $productId = (int)$item['id'];
                $qty       = (int)$item['qty'];

                /* ===============================
                   3️⃣ LOCK STOCK ROW (ANTI RACE CONDITION)
                =============================== */
                $stockRow = Database::fetchOne(
                    "SELECT quantity 
                     FROM product_stocks
                     WHERE product_id = ?
                     AND branch_id = ?
                     FOR UPDATE",
                    [$productId, $branchId]
                );

                if (!$stockRow) {
                    throw new \Exception("Stock record missing for product {$productId}");
                }

                $physicalStock = (int)$stockRow['quantity'];

                /* ===============================
                   4️⃣ CHECK IF FROM SUSPEND
                =============================== */
                if ($suspendId) {

                    // Check reserved quantity only for THIS suspend
                    $reserved = Database::fetchOne(
                        "SELECT COALESCE(SUM(quantity),0) as total
                         FROM stock_movements
                         WHERE product_id = ?
                         AND branch_id = ?
                         AND type = 'reserve'
                         AND reference_type = 'suspend'
                         AND reference_id = ?
                         AND reversed = 0",
                        [$productId, $branchId, $suspendId]
                    );

                    $reservedQty = (int)$reserved['total'];

                    if ($reservedQty < $qty) {
                        throw new \Exception("Reserved stock mismatch for product {$productId}");
                    }

                    $stockService->reverseSuspendReserve(
    $productId,
    $branchId,
    $suspendId
);

                }
                else {

                    // Normal sale → check available stock
                    $available = $stockService->getAvailableStock($productId, $branchId);

if ($available < $qty) {
$available = $physicalStock; // already locked

throw new \Exception(
    "Stock not enough for product ID {$productId}. Available: {$available}"
);

}

                }

                /* ===============================
                   5️⃣ DEDUCT PHYSICAL STOCK
                =============================== */
                
                $stockService->deductStock(
                $productId,$branchId,$qty,$saleId);

            }

            Database::commit();

            // Clear suspend flag after successful checkout
            unset($_SESSION['restored_suspend_id']);

// ✅ LOG SUCCESS BEFORE RETURN
$audit->log(
    'checkout_success',
    $saleId,
    'sale',
    'Sale completed successfully'
);

            return [
    'success' => true,
    'sale_id' => $saleId
];

        } 
        
        
catch (\Throwable $e) {

    Database::rollback();

    $audit->log(
        'checkout_failed',
        null,
        null,
        $e->getMessage()
    );

    return [
        'success' => false,
        'message' => $e->getMessage()
    ];
}


    }
}
