<?php
namespace App\Core;

use PDO;
use PDOException;

class Database
{
    protected static ?PDO $pdo = null;

    protected static function connect(): void
    {
        if (self::$pdo instanceof PDO) {
            return;
        }

        $config = require __DIR__ . '/../../config/database.php';

        try {
            self::$pdo = new PDO(
                "mysql:host={$config['host']};dbname={$config['name']};charset=utf8mb4",
                $config['user'],
                $config['pass'],
                [
                    PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES   => false,
                ]
            );
        } catch (PDOException $e) {
            // Fail fast and loud — better than silent corruption
            die('Database connection failed.');
        }
    }

    /* ===========================
       READ HELPERS
       =========================== */

    public static function fetchAll(string $sql, array $params = []): array
    {
        self::connect();
        $stmt = self::$pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    public static function fetchOne(string $sql, array $params = []): ?array
    {
        self::connect();
        $stmt = self::$pdo->prepare($sql);
        $stmt->execute($params);
        $row = $stmt->fetch();
        return $row === false ? null : $row;
    }

    /* ===========================
       WRITE HELPERS
       =========================== */

    public static function execute(string $sql, array $params = []): bool
    {
        self::connect();
        $stmt = self::$pdo->prepare($sql);
        return $stmt->execute($params);
    }

    public static function lastInsertId(): string
    {
        self::connect();
        return self::$pdo->lastInsertId();
    }

    /* ===========================
       TRANSACTION HELPERS
       =========================== */

    public static function begin(): void
    {
        self::connect();
        self::$pdo->beginTransaction();
    }

    public static function commit(): void
    {
        if (self::$pdo?->inTransaction()) {
            self::$pdo->commit();
        }
    }

    public static function rollback(): void
    {
        if (self::$pdo?->inTransaction()) {
            self::$pdo->rollBack();
        }
    }
}
